#ifndef _MESSAGING_H
#define _MESSAGING_H

#include "UnityNP.h"
#include "Mutex.h"

namespace UnityPlugin
{
	struct MessageAttachment
	{
		int dataSize;
		void* data;
	};

	// Structure for retrieving an in-game data message params from the scripts.
	struct InGameDataMessage
	{
		const unsigned char* npID;			// NpID, contains data corresponding to the SceNpID structure, can be used as id to send to when messaging.
		int npIDSize;						// Size of the npID byte array.
		int messageID;
		int dataSize;
		void* data;
	};

	// Structure for passing message params from the scripts.
	class MsgRequest
	{
	public:
		const char* body;						// The body text of the message.
		void* data;								// Attached data.
		int dataSize;							// Size of the attached data in bytes.
		int expireMinutes;						// Message expire time in minutes.

		// PS4 only.
		int npIDCount;							// Number npIDs the user can select in the message dialog.
		const char* dataName;					// The data name.
		const char* dataDescription;			// The data description.
		const char* iconPath;					// The path to an icon, which needs to be displayed in the message.
	};

	PRX_EXPORT bool PrxMessagingIsBusy();
	PRX_EXPORT bool PrxMessagingSendInGameDataMessage(const unsigned char* npID, void* data, int dataSize);
	PRX_EXPORT bool PrxMessagingSendMessage(const MsgRequest* request);	//const char* body, int expireMinutes, void* data, int dataSize);
	PRX_EXPORT bool PrxMessagingSendGameInvite(const MsgRequest* request);	//const char* body, int expireMinutes, void* data, int dataSize);
	PRX_EXPORT bool PrxMessagingShowMessageDialog();
	PRX_EXPORT bool PrxMessagingShowInviteDialog();
	PRX_EXPORT bool PrxMessagingGetMessageAttachment(MessageAttachment* attachment);
	PRX_EXPORT bool PrxMessagingGetGameInviteAttachment(MessageAttachment* attachment);
	PRX_EXPORT bool PrxHasInGameDataMessage();
	PRX_EXPORT bool PrxGetFirstInGameDataMessage(InGameDataMessage* message);
	PRX_EXPORT bool PrxRemoveFirstInGameDataMessage();

	class InGameMessageFIFO
	{
	public:
		std::list<InGameDataMessage> m_FIFO;
		SimpleLock m_Lock;

		void Add(InGameDataMessage& gameDataMsg);
		bool HasData();
		bool GetFirst(InGameDataMessage* gameDataMsg);
		bool RemoveFirst();
	};

	class InGameMessagePendingFIFO
	{
	public:
		std::list<int> m_FIFO;
		SimpleLock m_Lock;

		void Add(int gameDataMsgID);
		bool HasData();
		bool GetFirst(int *gameDataMsgID);
		bool RemoveFirst();
	};

	class CachedMessaging
	{
		SimpleLock m_Lock;
		bool m_Busy;
		sce::Toolkit::NP::MessageData m_Message;
		sce::Toolkit::NP::Utilities::Future<sce::Toolkit::NP::MessageAttachment> m_FutureAttachment;
		sce::Toolkit::NP::MessageAttachment* m_SessionInviteMessageAttachment;
		sce::Toolkit::NP::MessageAttachment* m_GameInviteAttachment;
		sce::Toolkit::NP::MessageAttachment* m_MessageAttachment;
		InGameMessagePendingFIFO m_InGameDataMessagePendingFIFO;
		InGameMessageFIFO m_InGameDataMessageFIFO;

		static char sUnityMessageMagic[8];

		enum UnityPluginMessageType
		{
			MESSAGE_TYPE_GAME_INVITE,
			MESSAGE_TYPE_GAME_DATA,
		};

		struct UnityPluginMessageHeader
		{
			char magic[8];
			UnityPluginMessageType messageType;
			int dataSize;
		};

	public:
		CachedMessaging();
		~CachedMessaging();
		void InitMessage( void );

		bool ProcessEvent(const sce::Toolkit::NP::Event& event);
		bool ProcessPendingReceives();

		bool IsBusy();

		bool SendInGameDataMessage(const unsigned char* npID, void* data, int dataSize);
		bool SendMessage(const MsgRequest* request);	//const char* body, int expireMinutes, void* data, int dataSize);
		bool SendGameInvite(const MsgRequest* request);	//const char* body, int expireMinutes, void* data, int dataSize);
		bool ShowArrivedMessageDialog();
		bool ShowArrivedInviteDialog();
		bool GetMessageAttachement(MessageAttachment* attachement);
		bool GetGameInviteAttachement(MessageAttachment* attachement);
		bool HasInGameDataMessages();
		bool GetFirstInGameDataMessage(InGameDataMessage* message);
		bool RemoveFirstInGameDataMessage();
		void Update();

		sce::Toolkit::NP::MessageAttachment* GetSessionInviteAttachment();
		void DumpSessionInviteMessageAttachment(SceChar8* messageData);
	};

	extern CachedMessaging gMessaging;
}

#endif // _SIGNING_H
